const { UUID, UUIDV4, DECIMAL, STRING, TEXT, INTEGER, DATEONLY, ENUM } = require('sequelize');

module.exports = (sequelize, DataTypes) => {
	const Bookings = sequelize.define(
		'Bookings',
		{
			id: {
				type: UUID,
				defaultValue: UUIDV4,
				allowNull: false,
				primaryKey: true,
			},
			customer_id: {
				type: UUID,
				allowNull: false,
			},
			host_id: {
				type: UUID,
				allowNull: false,
			},
			listing_id: {
				type: UUID,
				allowNull: false,
			},
			booking_no: {
				type: STRING(20),
				allowNull: false,
			},
			check_in: {
				type: DATEONLY,
				allowNull: false,
			},
			check_out: {
				type: DATEONLY,
				allowNull: false,
			},
			pax: {
				type: INTEGER,
				allowNull: false,
				defaultValue: 1,
			},
			price_per_night: {
				type: DECIMAL(10, 2),
				allowNull: false,
				defaultValue: 0.0,
			},
			service_fee: {
				type: DECIMAL(10, 2),
				allowNull: false,
				defaultValue: 0.0,
			},
			add_ons_total: {
				type: DECIMAL(10, 2),
				allowNull: false,
				defaultValue: 0.0,
			},
			total_amount: {
				type: DECIMAL(10, 2),
				allowNull: false,
				defaultValue: 0.0,
			},
			payment_option: {
				type: ENUM('Pay Now', 'Installment'),
				allowNull: false,
			},
			amount_paid: {
				type: DECIMAL(10, 2),
				allowNull: false,
				defaultValue: 0.0,
			},
			payment_status: {
				type: ENUM('Fully Paid', 'Unpaid', 'Partially Paid'),
				allowNull: false,
				defaultValue: 'Unpaid',
			},
			balance: {
				type: DECIMAL(10, 2),
				allowNull: false,
				defaultValue: 0.0,
			},
			phone_no: {
				type: STRING(15),
				allowNull: false,
			},
			payment_method: {
				type: ENUM('Online', 'Cash', 'Debit/Credit'),
				allowNull: false,
			},
			message_to_host: {
				type: TEXT,
				allowNull: true,
			},
			user_image: {
				type: STRING,
				allowNull: true,
			},
			status: {
				type: ENUM('Done', 'Upcoming', 'Cancelled', 'Pending'),
				allowNull: false,
				defaultValue: 'Pending',
			},
		},
		{
			tableName: 'bookings',
			timestamps: true,
			hooks: {
				afterCreate: async (booking, options) => {
					try {
						const { HostNotifications } = sequelize.models;

						const message = `A new booking has been created. Check-in is scheduled for ${booking.check_in}.`;

						await HostNotifications.create({
							host_id: booking.host_id,
							booking_id: booking.id,
							notif_type: 'New Booking',
							message,
							status: 'Unread',
						});

						// console.log(`Notification for new booking (ID: ${booking.id}) created.`);
					} catch (error) {
						console.error('Error creating notification for new booking:', error.message);
					}
				},
			},
		},
	);

	Bookings.associate = (models) => {
		Bookings.belongsTo(models.Listings, {
			foreignKey: 'listing_id',
			as: 'listing',
		});

		Bookings.hasMany(models.BookingAddOns, {
			foreignKey: 'booking_id',
			as: 'addOns',
		});

		Bookings.belongsTo(models.Users, {
			foreignKey: 'customer_id',
			as: 'customers',
		});

		Bookings.belongsTo(models.Users, {
			foreignKey: 'host_id',
			as: 'host',
		});
	};

	return Bookings;
};
